/**

Description: Holds the parsed table informaation. And in the case of a PDF
parse takes care of rendering PDF elements.
Benjamin E. Coe. 2007

-------------------------------<br />
This library is provided free of charge by Benjamin E. Coe and PLink. Simply
be kind and reference your use of our library.
-------------------------------<br />
*/
package com.plink.plextile;
import com.plink.plextile.util.*;
import com.lowagie.text.*;
import com.lowagie.text.pdf.*;
import java.util.*;
import java.awt.Color;


//Used to represent a textile Chunk, for use by the PDF.
public class TextileTable extends AbstractTextileChunk{

	//Data.
	private int ccount=0;//Cell count.
	private int width=0;//Columns.
	private int height=0;//Rows.
	private int start;//At what point in the data space does this table begin?
	private int stop;//At what point in the data space does this table end?
	private TextileParser MyTextileParser=null;//A central textile parser which stores data.
	
	private ArrayList Rows=new ArrayList();
	private Row CurrentRow=new Row();
	
	private TextileBlock MetaBlock=null;//Table level attributes.
	private ArrayList Tokens=new ArrayList();//Stack used to keep track of td versus th tags.
	private boolean handled=false;//Has this Textile Table been placed in a renderer yet?
	
	//This internal class represents a single row of our table.
	private class Row{
		private int start;//Index of rows starting position.
		private int stop;//Index of rows stopping position.
		private TextileBlock MetaBlock=null;//Row level style.
		private ArrayList Cells=new ArrayList();//Array List of cells.
		
		//Get the total widt of the row in cells.
		public int getWidth(){
			int width=0;
			ArrayList Chunk=new ArrayList();
			for(int i=0;i<Cells.size();i++){
				Cell MyCell=(Cell)Cells.get(i);
				width+=MyCell.getWidth();
			}
			return(width);
		}
		
		//Get an array list of lists cells containing an array list of their element.
		public ArrayList getChunks(){
			ArrayList Chunk=new ArrayList();
			for(int i=0;i<Cells.size();i++){
				Cell MyCell=(Cell)Cells.get(i);
				Chunk.add(MyCell);
			}
			return(Chunk);
		}
		
		//Add a cell object to the table.
		public void add(Cell C)
		{
			Cells.add(C);
		}
		
		//Insert a chunk into the appropriate cell.
		public void addChunk(int position,AbstractTextileChunk NewChunk){
			for(int i=0;i<Cells.size();i++){
				Cell Chunk=(Cell)Cells.get(i);
				if(position>=Chunk.getStart()&&position<=Chunk.getStop()){
					Chunk.addChunk(NewChunk);
				}
			}
		}
		
		//Set the starting index of the table.
		public void setStart(int start){
			this.start=start;
		}
		
		//Set the ending index of the table.
		public void setStop(int stop){
			this.stop=stop;
		}
		
		//Does an event occur at this position in the table?
		public int isEvent(int position){
			if(position+1!=stop)
			for(int i=0;i<Cells.size();i++){
				Cell MyCell=(Cell)Cells.get(i);
				if(MyCell.getStop()==position){
					return(BlockRenderer.CELL_END);
				}
			}
			return(-1);
		}
		
		//Get any HTML that should currently be rendered at the given index.
		public String getHTML(int position){
			if(position+1!=stop)
			for(int i=0;i<Cells.size();i++){
				Cell MyCell=(Cell)Cells.get(i);
				if(MyCell.getStart()==position){
					String returnMe="";
					
					
					if(MyCell.getMeta().isHeader()){
						Tokens.add("h");
						returnMe+="h"+TextileParser.applyTagMeta(MyCell.getMeta());
					}else{
						Tokens.add("d");
						returnMe+="d"+TextileParser.applyTagMeta(MyCell.getMeta());
					}
					
					if(MyCell.getMeta().getColspan()>0){
						returnMe+=" colspan=\""+MyCell.getMeta().getColspan()+"\"";
					}
					
					if(MyCell.getMeta().getRowspan()>0){
						returnMe+=" rowspan=\""+MyCell.getMeta().getRowspan()+"\"";
					}
						
					return(returnMe);
				}
			}
			return("d");
		}
		
		//Get the row level html (class)[language]{style}
		public String getHTML(){
			if(MetaBlock!=null){
				return(TextileParser.applyTagMeta(MetaBlock));
			}
			return("");
		}
		
		//Get the starting index of this row.
		public int getStart(){
			return(start);
		}
		
		//Get the stopping index of this row.
		public int getStop(){
			return(stop);
		}
		
		//Set the row specific data.
		public void setMetaBlock(TextileBlock MetaBlock){
			this.MetaBlock=MetaBlock;
		}
		
		/**
		Check whether a given position is in this list.
		*/
		public boolean inTable(int position){
			for(int i=0;i<Cells.size();i++){
				Cell Chunk=(Cell)Cells.get(i);
				if(position>=Chunk.getStart()&&position<=Chunk.getStop()){
					return(true);
				}
			}
			return(false);
		}
		
		//Return the TextileBlock containing meta information.
		public TextileBlock getMeta(){
			return(MetaBlock);
		}
	}
	
	//This internal class represents a single cell in the table.
	private class Cell{
		private int start;//Index of this cells starting position in the data.
		private int stop;//Index of this cells stopping position in the data.
		private TextileBlock MetaBlock=null;//This Textile Block contains meta information.
		private ArrayList Chunks=new ArrayList();
		
		public Cell(int start,int stop,TextileBlock MetaBlock){
			this.start=start;
			this.stop=stop;
			this.MetaBlock=MetaBlock;
		}
		
		//Get the rowspan of this cell. NOT CURRENTLY IMPLEMENTED.
		public int getRowspan(){
			if(MetaBlock!=null)
				return(MetaBlock.getRowspan());
			return(0);
		}
		
		/*
		Get the column span. 
		*/
		public int getColspan(){
			if(MetaBlock!=null)
				return(MetaBlock.getColspan());
			return(0);
		}
		
		//How wide is this cell, takes into account colspan.
		public int getWidth(){
			int width=1;
			if(MetaBlock!=null)
				width+=MetaBlock.getColspan();
			return(width);
		}
		
		//Return all the Chunks currently contained within this cell.
		public ArrayList getChunks(){
			return(Chunks);
		}
		
		//Ad a chunk to this cell.
		public void addChunk(AbstractTextileChunk Chunk){
			Chunks.add(Chunk);
		}
		
		//Get the starting index where this cell begins.
		public int getStart(){
			return(start);
		}
		
		//Get the stopping position within the data where this cell ends.
		public int getStop(){
			return(stop);
		}
		
		//Get the TextileBlock representing cell meta information.
		public TextileBlock getMeta(){
			return(MetaBlock);
		}
	}
	
	//Constrructor.
	public TextileTable(TextileParser MyTextileParser){
		this.MyTextileParser=MyTextileParser;
	}	
	
	public void setMeta(TextileBlock MetaBlock){
		this.MetaBlock=MetaBlock;
	}
	
	/**
	Add a Chunk (PDF element) to a cell in this table at the
	given position.
	*/
	public void addChunk(int position,AbstractTextileChunk NewChunk){
		for(int i=0;i<Rows.size();i++){
			Row Chunk=(Row)Rows.get(i);
			if(Chunk.inTable(position))
				Chunk.addChunk(position,NewChunk);
		}
	}
	
	/**
	Generate a PDF element based on the ArrayLists making up this
	table's structure.
	*/
	public Element getChunk(){ 

		try{
		
			if(Rows.size()==0)
				return(null);
			Row Temp=(Row)Rows.get(0);
				
			com.lowagie.text.Table table = new com.lowagie.text.Table(Temp.getWidth());
			table.setWidth(98.0f);
			table.setPadding(2.0f);
			
			//For each row.
			for(int i=0;i<Rows.size();i++){
				Row RowChunk=(Row)Rows.get(i);
				
				for(int iii=0;iii<RowChunk.getChunks().size();iii++){//Fetch an array list of cells.
				
				Cell Chunk=(Cell)RowChunk.getChunks().get(iii);
				com.lowagie.text.Cell tempc =  new com.lowagie.text.Cell();
				if(Chunk.getColspan()>0)//Cet the cell column span.
					tempc.setColspan(Chunk.getColspan());
				
				for(int ii=0;ii<Chunk.getChunks().size();ii++){//For each cell output the array list of chunks.
				
				
					AbstractTextileChunk temp=(AbstractTextileChunk)Chunk.getChunks().get(ii);
					if(temp!=null){
						if(temp.getChunk()!=null){
							tempc.addElement((Element)temp.getChunk());
						}
					}
					
				}
				
				table.addCell(tempc);
				
				}
			}
			
			return(table);
		
		}catch(Exception e){
			e.printStackTrace();
		}
		return(null);
	}
	
	/**
	Set the starting index of this table.
	*/
	public void setStart(int start){
		this.start=start;
	}
	
	/**
	Set the stopping index of this table.
	*/
	public void setStop(int stop){
		this.stop=stop;
	}
	
	/**
	Get the starting index of this table.
	*/
	public int getStart(){
		return(start);
	}
	
	/**
	Get the stopping index of this table.
	*/
	public int getStop(){
		return(stop);
	}
	
	/**
	Stores the last HTML generated based on list position
	events. Used by a TextParser to output this table's data.
	*/
	public String currentHTML="";
	public String getHTML(){
		return(currentHTML);
	}
	
	/**
	Add a cell to the table. cell has the starting and stopping positions
	provided in the data's string. The MetaBlock represents cell level
	meta inforamtion.
	*/
	public void addCell(int start,int stop,TextileBlock MetaBlock){
		width++;
		CurrentRow.add(new Cell(start,stop,MetaBlock));
	}
	
	/**
	Set the style of the current row.
	Checks the information to see whether it also provides table style.
	*/
	public void setRowStyle(String style){
		
		//Get rid of left over | characters.
		style=style.replaceAll("|","")+" ";
		if(style.charAt(0)=='\n')
			style=style.substring(1,style.length());
		
		if(height==0){//This is our first style inforation, try to parse table style.
			String parseMe[]=style.split("\n");
			if(parseMe.length==1){
				String Temp=null;
				if(style.charAt(0)!='\n'){
					Temp=MyTextileParser.parseAttribComplex(0,parseMe[0]);
				}else{
					Temp=MyTextileParser.parseAttribComplex(1,parseMe[0]);
				}
				if(Temp!=null)
					Temp=Temp.substring(0,Temp.length()-1);
				
				TextileBlock MetaInfo=new TextileBlock();
				if(Temp!=null){
					MetaInfo.parseAttrib(Temp+" ");
				}
				CurrentRow.setMetaBlock(MetaInfo);
			}else{
			
				String Temp=null;
				if(style.charAt(0)!='\n'){
					Temp=MyTextileParser.parseAttribComplex(0,parseMe[1]);
				}else{
					Temp=MyTextileParser.parseAttribComplex(1,parseMe[1]);
				}
				if(Temp!=null)
					Temp=Temp.substring(0,Temp.length()-1);
				
				TextileBlock MetaInfo=new TextileBlock();
				if(Temp!=null){
					MetaInfo.parseAttrib(Temp+" ");
				}
				CurrentRow.setMetaBlock(MetaInfo);
				
				if(style.charAt(0)!='\n'){
					Temp=MyTextileParser.parseAttribComplex(5,parseMe[0]);
				}else{
					Temp=MyTextileParser.parseAttribComplex(6,parseMe[0]);
				}
				if(Temp!=null)
					Temp=Temp.substring(0,Temp.length()-1);
				
				MetaBlock=new TextileBlock();
				if(Temp!=null){
					MetaBlock.parseAttrib(Temp+" ");
				}
			}
		}else{//This isn't the first data that's been fetched, don't try to parse table meta inforamtion.
			String Temp=null;
			if(style.charAt(0)!='\n'){
				Temp=MyTextileParser.parseAttribComplex(0,style);
			}else{
				Temp=MyTextileParser.parseAttribComplex(1,style);
			}
			if(Temp!=null)
				Temp=Temp.substring(0,Temp.length()-1);
			
			TextileBlock MetaInfo=new TextileBlock();
			if(Temp!=null){
				MetaInfo.parseAttrib(Temp+" ");
			}
			CurrentRow.setMetaBlock(MetaInfo);
		}
	}
	
	/**
	Add a row to the table.
	*/
	public void addRow(int start,int stop){
		height++;
		CurrentRow.setStart(start);
		CurrentRow.setStop(stop);
		Rows.add(CurrentRow);
		CurrentRow=new Row();
	}
	
	/**
	Return a reference to this tables meta data block.
	*/
	public TextileBlock getBlock(){
		return(null);
	}
	
	/**
	Get a string representing this table's
	meta information.*/
	public String getMeta(){
		if(MetaBlock!=null){
			return(TextileParser.applyTagMeta(MetaBlock));
		}
		return("");
	}
	
	/**
	Check whether a table event should be generated. Using the style information
	from the Table, Cells, and Rows. the current HTML string is set and can be fetched
	by a text parser.
	*/	
	public int isEvent(int position){
		if(position==start){//Table start event.
			Row MyRow=(Row)Rows.get(0);
			currentHTML="<table"+getMeta()+"><tr"+MyRow.getHTML()+"><t"+MyRow.getHTML(position)+">";
			return(BlockRenderer.TABLE_START);
		}
		if(position==stop){//Table stop event.
			if(Tokens.size()>0)
				currentHTML="</t"+(String)Tokens.get(Tokens.size()-1)+"></tr></table>";
			else
				currentHTML="</td></tr></table>";
			return(BlockRenderer.TABLE_END);
		}
		
		if(position+1!=stop)
		for(int i=0;i<Rows.size();i++){
			Row MyRow=(Row)Rows.get(i);
			if(MyRow.getStop()==position){
				Row TempRow=null;
				if(i<Rows.size()-1)
					TempRow=(Row)Rows.get(i+1);
				if(TempRow!=null){
					if(Tokens.size()>0){
						currentHTML="</t"+(String)Tokens.get(Tokens.size()-1)+"></tr><tr"+TempRow.getHTML()+"><t"+TempRow.getHTML(position+1)+">";
					}else{
						currentHTML="</td></tr><tr"+TempRow.getHTML()+"><t"+TempRow.getHTML(position+1)+">";
					}
				}else{
					if(Tokens.size()>0)
						currentHTML="</t"+(String)Tokens.get(Tokens.size()-1)+"></tr><tr><td>";
					else
						currentHTML="</td></tr><tr><td>";
				}
				return(BlockRenderer.ROW_END);
			}
			
			if(MyRow.isEvent(position)>0){
				if(Tokens.size()>0)
					currentHTML="</t"+(String)Tokens.get(Tokens.size()-1)+"><t"+MyRow.getHTML(position+1)+">";
				else
					currentHTML="</td><t"+MyRow.getHTML(position+1)+">";
				return(BlockRenderer.CELL_END);
			}
		}
		
		
		return(-1);
	}
	
	/**
	Check whether a given position is in this list.
	*/
	public boolean inTable(int position){
		for(int i=0;i<Rows.size();i++){
			Row Chunk=(Row)Rows.get(i);
			if(Chunk.inTable(position)){
				return(true);
			}
		}
		return(false);
	}
	
	/**
	Has this Chunk been dealt with by a parent yet.
	*/
	public boolean isHandled(){
		return(handled);
	}
	
	/**
	Set whether the chunk has been handled.
	*/
	public void setHandled(boolean handled){
		this.handled=handled;
	}
	
	/**
	toString()
	*/
	public String toString(){
		return("");
	}
}