/**

This class performs parsing for the various syntax defined in a
textile data chunk. It also applies many of the text tranforamtion steps required.<br />
This class contains parsing methods that are common to both ParsePDF and ParseText.
It also serves as a location to register central information for specific types of elements, e.g.,
Footers, and Aliases.

Benjamin E. Coe. (c)2007<br />

-------------------------------<br />
The Plextile library is provided free of charge by Benjamin E. Coe and the PLink team. Simply
be kind and reference your use of our library.
-------------------------------<br />
caminatored
*/
package com.plink.plextile;
import  com.plink.plextile.util.*;
import java.util.*;
import java.net.URL;

public class TextileParser{
	private String freedata="";//Data that next parsing run should be applied to.
	//Represents known links.
	private ArrayList Destination=new ArrayList();
	private ArrayList AttribList=new ArrayList();//Array of tag attributes.
	private ArrayList SkipList=new ArrayList();
	private TextileParser ExternalParser=null;
	
	/**
	Register an external parser for global events.
	*/
	public void setExternalParser(TextileParser ExternalParser){
		this.ExternalParser=ExternalParser;
	}
	
	/**
	Check whether a destination exists. (Either in document or an alias.)
	this is used by alias links, and by footnotes.
	*/
	public boolean destinationExists(String check){
		check=check.toUpperCase();
		for(int i=0;i<Destination.size();i++){
			String temp=((String[])Destination.get(i))[0];
			temp=temp.toUpperCase();
			if(temp.equals(check))
				return(true);
		}	
		return(false);
	}
	
	/**
	Fetch the URL of the alias that has been registered.
	*/
	public String getDestination(String check){
		check=check.toUpperCase();
		for(int i=0;i<Destination.size();i++){
			String temp[]=((String[])Destination.get(i));
			temp[0]=temp[0].toUpperCase();
			if(temp[0].equals(check)){
				return(temp[1]);
			}
		}	
		return("");
	}
	
	/**
	Register a destination in the document. (Alias or in document link)
	*/
	public synchronized void addDestination(String des[]){
		Destination.add(des);
	}
	
	/**
	Register a position in the parsing process where we should skip characters.
	Used by many of the helper functions to specify portions of the data
	space that should be skipped when creating the final text data.
	*/
	public void addSkip(int skip[]){
		SkipList.add(skip);
	}
	
	/**
	Reset the skip values for a new block. (Used when doing a non-threaded parse).
	*/
	public void resetSkip(){
		SkipList=new ArrayList();
	}
	
	/**
	Check for skip.
	*/
	public int getSkip(int position){
		for(int i=0;i<SkipList.size();i++){
			int Check[]=(int[])SkipList.get(i);
			if(position==Check[0]){
				return(Check[1]);
			}
		}
		return(0);
	}
	
	/*
	Register an attribute block at the given position in the document space.
	This is used to apply specific meta information to tags {^,@,*,etc}
	*/
	protected void registerAttrib(int position,String parsestyle){
		TextileBlock Block=new TextileBlock();
		Block.parseAttrib(parsestyle);
		TagMetaData TMD=new TagMetaData(Block,position,parsestyle.length()-1);
		AttribList.add(TMD);
	}
	
	/**
	Check whether a given index has an attribute block associated with it.
	*/
	public TagMetaData hasAttrib(int position){
		for(int i=0;i<AttribList.size();i++){
			TagMetaData TMD=(TagMetaData)AttribList.get(i);
			if(TMD.getPosition()==position){
				return(TMD);
			}
		}
		return(null);
	}
	
	/**
	freedata represents a string without the tagging reserved characters... which are replaced with whitespace.
	this value is set during the ParseText phase.
	*/
	public String getTagFreeData(){
		return(freedata);
	}
	
	/**
	Check for charcters that have been escaped, and replace them
	before the parsing process takes place.
	*/
	public String applyEscape(String data){
		boolean on=true;
		
		int sb1=0;
		StringBuffer SB=new StringBuffer(data.length());
				
		for(int i=0;i<data.length();i++){
			char c=data.charAt(i);
		
			if(findString(i,c,data,"\\&quot;")!=null){
				SB.insert(sb1,"&#"+((int)'"') +";");
				sb1+=5;
				i+=6;
			}else
			
			if(c=='\\'&&on&&i!=data.length()-1&&!isAlphaNumeric(data.charAt(i+1))){
				on=false;
				continue;
			}else
			
			if(!on){
				String temp="&#"+((int)c)+";";
				SB.insert(sb1,temp);
				sb1+=temp.length();
				on=true;
			}

			else{
				SB.insert(sb1,c);
				sb1++;
			}
		}
		return(SB.substring(0,sb1));
	}
	
	/**
	Replace the various special characters '...','(c), etc.
	*/
	private int templength=0;
	public String replaceSpecial(String tempdata){
	
		int sb1=0;
		StringBuffer SB=new StringBuffer(tempdata.length());
		String temps;
	
		for(int i=0;i<tempdata.length();i++){
			char c=tempdata.charAt(i);
			
			if(findString(i,c,tempdata,"...")!=null){
				temps="&#8230;";
				SB.insert(sb1,temps);
				sb1+=temps.length();
				i+=2;
			}else
			
			if(findString(i,c,tempdata," - ")!=null){
				temps=" &#8211; ";
				SB.insert(sb1,temps);
				sb1+=temps.length();
				i+=2;
			}else
			
			if(findString(i,c,tempdata," x ")!=null){
				temps=" &#215; ";
				SB.insert(sb1,temps);
				sb1+=temps.length();
				i+=2;
			}else
			
			if(findString(i,c,tempdata,"--")!=null){
				temps="&#8212;";
				SB.insert(sb1,temps);
				sb1+=temps.length();
				i+=1;
			}
			
			else
			
			if(findString(i,c,tempdata,"(tm)")!=null){
				temps="&#8482;";
				SB.insert(sb1,temps);
				sb1+=temps.length();
				i+=3;
			}else
			
			if(findString(i,c,tempdata,"(r)")!=null){
				temps="&#174;";
				SB.insert(sb1,temps);
				sb1+=temps.length();
				i+=2;
			}else
			
			if(findString(i,c,tempdata,"(c)")!=null){
				temps="&#169;";
				SB.insert(sb1,temps);
				sb1+=temps.length();
				i+=2;
			}
			
			else{
				SB.insert(sb1,c);
				sb1++;
			}
		}
		templength=sb1;
		return(SB.substring(0,sb1));
	}
	
	/**
	Replace a few specific characters before parse starts.
	*/
	public String initialEscape(String tempdata){
		StringBuffer SB=new StringBuffer(tempdata.length());
		int ii=0;
		for(int i=0;i<tempdata.length();i++){
			char c=tempdata.charAt(i);
			
			/*if(findString(i,c,tempdata,"&amp;")!=null){//Make sure '&' isn't escaped
				SB.insert(ii,'&');
				ii++;
				i+=4;
			}else*/
			
			if(findString(i,c,tempdata,"&#"+((int)'\n')+";")!=null){//Replace escaped '\n's with \n.
				SB.insert(ii,'\n');
				ii++;
				i+=4;
			}else
			
			if(findString(i,c,tempdata,"&#13;")!=null){//Get rid of \r characters.
				i+=4;
			}else{
				SB.insert(ii,c);
				ii++;
			}
				
		}
		
		return(SB.substring(0,ii));
	}

	/**
	Performs the initial Phrase Modifier parse of the textile data.
	Returns an ArrayList consisting of ranges that a tag applies to.
	Also applies some transformations (r)(c)(tm), and other priming steps.
	*/
	public synchronized ArrayList ParseTextile(String parseMe,ComplexBlockRenderer Observer){
	
		ComplexBlockRenderer ListChecker=new ComplexBlockRenderer(null,this,true);//Used to check for list tokens.
		ArrayList DataList=getParseList(parseMe+" ");
		StringBuffer SB=null;
		StringBuffer SB2=null;
				
		//Our tags.
		char tags[]={'*','_','-','+','^','~','@','%','>','<',(char)127};
		ArrayList tagranges=new ArrayList();
		String returnMe="";

		//Quickly partition the data so that <code> tags are ignored, checks for List starting tags aswell.
		String temps="";
		int TogglePoints[]=new int[DataList.size()];
		int skip=0;
		int Result[]=new int[2];
		int sb1=0;
		SB=new StringBuffer(parseMe.length());
		for(int i=0;i<DataList.size();i++){
			
			if(i%2==0){
			
				//Check for escaped characters.
				String escape=applyEscape((String)DataList.get(i));
				
				//Fix an issue regarding token overlap and two character tokens.
				skip=0;
				for(int fix=0;fix<escape.length();fix++){
					char c=escape.charAt(fix);
					
					if(fix<	escape.length()-1&&c=='?'&&escape.charAt(fix+1)=='?'&&skip==0){
						SB.insert(sb1,(char)127);
						sb1++;
						
						fix++;
					}
					else if(fix<escape.length()-1&&c=='_'&&escape.charAt(fix+1)=='_'&&skip==0){
						SB.insert(sb1,'<');
						sb1++;
						fix++;
					}else if(fix<escape.length()-1&&c=='*'&&escape.charAt(fix+1)=='*'&&skip==0){
						if(fix==0){
							if((Result=ListChecker.checkForToken(fix,escape))==null){
								SB.insert(sb1,'<');
								sb1++;
								fix++;
							}else{
								SB.insert(sb1,c);
								sb1++;
								skip=Result[0]+1;
							}
						}else{
							if((Result=ListChecker.checkForToken(fix-1,escape))==null){
								SB.insert(sb1,'>');
								sb1++;
								fix++;
							}else{
								SB.insert(sb1,c);
								sb1++;
								skip=Result[0]+1;
							}
						}
					}else{
						SB.insert(sb1,c);
						sb1++;
					}
					
					if(skip>0)
						skip--;
				}
				
				//Replace various special characters.
				SB.insert(0,replaceSpecial(SB.substring(0,sb1)));
				sb1=templength;
			}else{
				String freedata=(String)DataList.get(i);
				//Make sure that HTML elements within a code tag are properly escaped.
				if(findString(0,freedata.charAt(0),freedata,"&lt;code&gt;")!=null&&freedata.length()>25){
					freedata=freedata.substring(12,freedata.length()-13);
					freedata=freedata.replaceAll("&gt;","&#"+((int)'>')+";");
					freedata=freedata.replaceAll("&lt;","&#"+((int)'<')+";");
					freedata="&lt;code&gt;"+freedata+"&lt;/code&gt;";
				}
				SB.insert(sb1,freedata);
				sb1+=freedata.length();
			}
			
			if(i==0){
				TogglePoints[i]=sb1;
			}else
				TogglePoints[i]=sb1+TogglePoints[i-1];
				
		}
		freedata=SB.substring(0,sb1);
		
		//Split the text based on the tags.
		boolean on=true;
		temps="";
		skip=0;
		
		SB=new StringBuffer(freedata.length());
		
		//Build up a list of positions in the data where tags should be applied.
		for(int i=0;i<freedata.length();i++){
			char d=freedata.charAt(i);
		
			//Check whether we're in a code block.
			for(int fix=0;fix<TogglePoints.length;fix++){
				if(TogglePoints[fix]==i){
					if(on){
						on=false;
					}else{
						on=true;
					}
				}
			}
			
			//Check for tags and register them in the data space.
			boolean foundTag=false;
			
			//Peform various other parsing tasks that have been consolidated into this one pass.
			boolean inTable=false;
			if(on){	
				inTable=Observer.buildComplexElements(d,i,freedata);//Build tables and lists.
				if(parseLinks(d,i,freedata,false)){//Check for links. IMG,A,ACRONYM,FN
					if(i>0){
						parseLinks(freedata.charAt(i-1),i-1,freedata,true);
						parseLinks(d,i,freedata,true);
					}
				}
			}
			
			
			if(on&&skip==0)
			for(int ii=0;ii<tags.length;ii++){
			
				//Fix the * token conflict with lists.
				if(ii==0&&freedata.charAt(i)==tags[ii]){
					if(i==0){
						if((Result=ListChecker.checkForToken(i,freedata))!=null){
							skip+=Result[0]+1;
							break;
						}
					}else if((Result=ListChecker.checkForToken(i-1,freedata))!=null){
						skip+=Result[0]+1;
						break;
					}
				}
				
				if(d==tags[ii]){
					if(!(i==0&&d=='^'))
					if(i!=freedata.length()-1)
					if(i==0||(freedata.charAt(i-1)=='\n'&&!inTable)||((freedata.charAt(i+1)!=' ')&&validTagStart(freedata.charAt(i-1)))){
					
						//Make sure we can't have two tags with nothing inbetween.
						if(freedata.charAt(i+1)==d||(i>0&&freedata.charAt(i-1)==d))
							break;
					
						boolean closed=true;
						for(int index=0;index<tagranges.size();index++){
							int ranges[][]=(int[][])tagranges.get(index);
							if(ranges[0][1]==i){
								closed=false;
							}
						}
						
						if(closed){
							closed=false;
							int ranges[][]=new int[2][2];
							ranges[1][0]=ii;
							ranges[0][0]=i+1;
							int offset=0;
							
							boolean fetchAttrib=false;
							boolean noAttrib=false;
							char stopToken=' ';
							String parseAttrib="";
							String tempAttrib="";
							int tokens=0;
							for(int iii=i+1;iii<freedata.length();iii++){
								offset++;
								//Fetch the attribute string.
								if(fetchAttrib)
									tempAttrib+=freedata.charAt(iii);
								
								//Parse out a tags attribute data.
								if(!fetchAttrib&&!noAttrib){
									char c=freedata.charAt(iii);
									if(c=='('||c=='['||c=='{'){
										fetchAttrib=true;
										parseAttrib+=c;
										if(c=='(')
											stopToken=')';
										if(c=='{')
											stopToken='}';
										if(c=='[')
											stopToken=']';
									}else
										noAttrib=true;
								}
								
								if(fetchAttrib&&freedata.charAt(iii)==stopToken){
									fetchAttrib=false;
									if(iii!=freedata.length()-1&&freedata.charAt(iii+1)!=tags[ii]){
										tokens++;
										parseAttrib+=tempAttrib;
									}
									tempAttrib="";
								}
								
								if(freedata.charAt(iii)==tags[ii]){
									closed=true;
									break;
								}
							}
							
							if(closed){
								foundTag=true;
								
								if(tokens>0){
									registerAttrib(i+1,parseAttrib+".");
								}
							
								ranges[0][1]=i+offset;
								tagranges.add(ranges);
							}
						}
						
					}
				}
			}
			
			//Fix the extra character problem with two character tags.
			if(!on||!foundTag||(d!='>'&&d!='<'&&d!=((char)127)))
				SB.insert(i,d);
			else
				SB.insert(i,' ');
				
			if(skip>0)
				skip--;
		}
		freedata=SB.toString();
		
		//Order our tags by position of start.
		ArrayList temp=new ArrayList();
		for(int i=0;i<tagranges.size();i++){
			int ranges[][]=(int[][])tagranges.get(i);
			if(temp.size()==0)
				temp.add(ranges);
			else{
				boolean add=false;
				for(int ii=0;ii<temp.size();ii++){
					int ranges2[][]=(int[][])temp.get(ii);
					int r1=ranges[0][0];
					int r2=ranges2[0][0];
					
					if(r1<r2){
						temp.add(ii,ranges);
						add=true;
						break;
					}
				}
				if(!add)
					temp.add(ranges);
			}
		}
		
		return(temp);
	}
	
	/**
	Parse a simple attribute string from a given token.[EN](CLASS){STYLE}
	*/
	public String parseAttrib(int position,String freedata){
		boolean fetchAttrib=false;
		boolean noAttrib=false;
		char stopToken=' ';
		String parseAttrib="";
		String tempAttrib="";
		int tokens=0;
		for(int iii=position;iii<freedata.length();iii++){
			//Fetch the attribute string.
			if(fetchAttrib)
				tempAttrib+=freedata.charAt(iii);
			
			//Parse out a tags attribute data.
			if(!fetchAttrib&&!noAttrib){
				char c=freedata.charAt(iii);
				if(c=='('||c=='['||c=='{'){
					fetchAttrib=true;
					parseAttrib+=c;
					if(c=='(')
						stopToken=')';
					if(c=='{')
						stopToken='}';
					if(c=='[')
						stopToken=']';
				}else
					noAttrib=true;
			}
			
			if(fetchAttrib&&freedata.charAt(iii)==stopToken){
				fetchAttrib=false;
				if(iii!=freedata.length()-1){
					tokens++;
					parseAttrib+=tempAttrib;
				}
				tempAttrib="";
			}
		}
		if(tokens==0)
			return(null);
		return(parseAttrib+".");
	}
	
	/**
	Parse a complex attribute string from a given token.>)^[EN](CLASS){STYLE}.
	*/
	public String parseAttribComplex(int position,String freedata){
		boolean fetchAttrib=false;
		boolean noAttrib=false;
		char stopToken=' ';
		String parseAttrib="";
		String tempAttrib="";
		int tokens=0;
		for(int iii=position;iii<freedata.length();iii++){
			//Fetch the attribute string.
			tempAttrib+=freedata.charAt(iii);
			
			//Parse out a tags attribute data.
			if(!fetchAttrib){
				char c=freedata.charAt(iii);
				if(c=='('||c=='['||c=='{'){
					fetchAttrib=true;
					if(c=='(')
						stopToken=')';
					if(c=='{')
						stopToken='}';
					if(c=='[')
						stopToken=']';
				}
			}
			
			if(fetchAttrib&&freedata.charAt(iii)==stopToken){
				fetchAttrib=false;
			}
			
			if(!fetchAttrib&&freedata.charAt(iii)==' ')
				break;
			
			if(freedata.charAt(iii)=='.'){
				tokens=1;
				break;
			}
		}
		if(tokens==0)
			return(null);
			
		return(tempAttrib);
	}
	
	/**
	Find an occurence of the <CODE> tag so that we can avoid parsing this block.
	*/
	protected String codeTagCheck(int position,String data){
		if(position+13>data.length())
			return(null);
			
		if(data.substring(position,position+12).equals("&lt;code&gt;"))
			return(data.substring(position,position+12));
			
		if(data.substring(position,position+13).equals("&lt;/code&gt;"))
			return(data.substring(position,position+13));
			
		return(null);
	}
	
	/**
	Find an occurence of the &quot; tag.
	*/
	protected String quoteTagCheck(int position,String data){
		if(position+6>data.length())
			return(null);
			
		if(data.substring(position,position+6).equals("&quot;"))
			return(data.substring(position,position+6));
			
		return(null);
	}
	
	/**
	Find an occurence of the given string.
	*/
	public static String findString(int position,char c,String data,String match){
		if(c!=match.charAt(0))
			return(null);
	
		if(match.length()+position>data.length())
			return(null);
			
		String temp=data.substring(position,position+match.length());
		if(temp.equals(match))
			return(temp);
			
		return(null);
	}
	
	/**
	Find an occurence of the given string. (Ignoring case)
	*/
	public String findStringUpper(int position,String data,String match){
		if(match.length()+position>data.length())
			return(null);
			
		String temp=data.substring(position,position+match.length());
		if(temp.toUpperCase().equals(match.toUpperCase()))
			return(temp);
			
		return(null);
	}
	
	/**
	Final replacement. This should be run before text is rendered to change back tokens
	that were temporarily changed. (USED MAINLY IN PDF CREATION).
	*/
	public String finalReplacement(String tempdata){
		StringBuffer temps=new StringBuffer(tempdata.length());
		for(int i=0;i<tempdata.length();i++){
			char c=tempdata.charAt(i);
		
			if(c=='>')
				temps.append("**");
			else
			
			if(c=='<')
				temps.append("__");
				
			else
			
			if(c==(char)127)
				temps.append("??");
		
			else
			
			if(findString(i,c,tempdata,"&amp;")!=null){
				temps.append('&');
				i+=4;
			}else
			
			if(findString(i,c,tempdata,"&quot;")!=null){
				temps.append("\"");
				i+=5;
			}else
			
			if(findString(i,c,tempdata,"&lt;")!=null){
				temps.append("<");
				i+=3;
			}else
			
			if(findString(i,c,tempdata,"&gt;")!=null){
				temps.append(">");
				i+=3;
			}else
				temps.append(c);
		}
		return(temps.toString());
	}
	
	/**
	Find occurences of HTML tags so that we can turn off the parser briefly.
	*/
	String Tags[]={"&LT;IMG","&LT;P","&LT;A","&LT;SPAN","&LT;B","&LT;I","&LT;H"};
	protected String checkTags(int position,String data){
		for(int i=0;i<Tags.length;i++){
			if(!(position+Tags[i].length()>data.length())){
				if(data.substring(position,position+Tags[i].length()).toUpperCase().equals(Tags[i]))
					return(data.substring(position,position+Tags[i].length()));
			}
		}
		return(null);
	}
	
	/**
	Return the data split along blocks that should not have tags applied.
	This is used both for quote conversion and for the actual tagging.
	*/
	protected ArrayList getParseList(String freedata){
		ArrayList DataList=new ArrayList();
		//Make sure we don't effect code blocks or data within tags.
		int code=0;
		int bracket=0;
		
		int finish=0;
		StringBuffer SB=new StringBuffer(freedata.length());
		
		for(int i=0;i<freedata.length();i++){
			char c=freedata.charAt(i);
		
			if(code==0&&bracket==0&&(checkTags(i,freedata)==null&&codeTagCheck(i,freedata)==null)){
				SB.insert(finish,c);
				finish++;
			}else{
			
				if(code==0&&bracket==0){
					DataList.add(SB.substring(0,finish));
					finish=0;
				}
				String add="";
				
				if(bracket==0)
				if((add=codeTagCheck(i,freedata))!=null){
					if(code==0){
						i+=add.length()-1;
						code++;
						
						SB.insert(finish,add);
						finish+=add.length();
		
						continue;
					}else{
						i+=add.length()-1;
						code=0;
						
						SB.insert(finish,add);
						finish+=add.length();
		
						DataList.add(SB.substring(0,finish));
						finish=0;
						continue;
					}
				}
				
				if(code>=1){
					SB.insert(finish,c);
					finish++;
				}
				
				if(code==0)
				if((add=checkTags(i,freedata))!=null||(add=findStringUpper(i,freedata,"&gt;"))!=null){
					if(bracket==0){
						i+=add.length()-1;
						bracket++;
						SB.insert(finish,add);
						finish+=add.length();
						continue;
					}else{
						i+=add.length()-1;
						bracket=0;
						
						SB.insert(finish,add);
						finish+=add.length();
						DataList.add(SB.substring(0,finish));
						finish=0;
						
						continue;
					}
				}
				
				if(bracket>=1){
					SB.insert(finish,c);
					finish++;
				}
			}
		}
		DataList.add(SB.substring(0,finish));
		return(DataList);
	}
	
	/**
	Primes the data so that <CODE> and bc. blocks aren't split into multiple blocks.
	*/
	public String makeCodeSafe(String freedata){
		ArrayList DataList=new ArrayList();
		//Make sure we don't effect code blocks or data within tags.
		int code=0;
		int finish=0;
		StringBuffer SB=new StringBuffer(freedata.length());
		
		for(int i=0;i<freedata.length();i++){
			char c=freedata.charAt(i);
		
			if(code==0&&codeTagCheck(i,freedata)==null){
				SB.insert(finish,c);
				finish++;
			}else{
				if(code==0){
					DataList.add(SB.substring(0,finish));
					finish=0;
				}
				String add=null;
				
				if((add=codeTagCheck(i,freedata))!=null){
					if(code==0){
						i+=add.length()-1;
						code++;
						SB.insert(finish,add);
						finish+=add.length();
						continue;
					}else{
						i+=add.length()-1;
						code=0;
						SB.insert(finish,add);
						finish+=add.length();
						DataList.add(SB.substring(0,finish));
						finish=0;
						continue;
					}
				}
				
				if(freedata.charAt(i)!='\n'){
					SB.insert(finish,freedata.charAt(i));
					finish++;
				}else{
					SB.insert(finish,"&#"+((int)'\n')+";");
					finish+=5;
				}
			}
		}
		DataList.add(SB.substring(0,finish));
		
		//Rebuild the data string.
		String ts="";
		for(int i=0;i<DataList.size();i++){
			String s=(String)DataList.get(i);
			ts+=s;
		}
		return(ts);
	}

	/**
	Convert the " and ' in the Textile data to their alternate
	representations.
	*/
	public String quoteConversion(String data){
		ArrayList DataList=getParseList(data);
		String temps="";
		
		StringBuffer SB=new StringBuffer(data.length());
		int sb1=0;
		
		for(int index=0;index<DataList.size();index++){
			//Add curly double quotes.
			String freedata=(String)DataList.get(index);
		
			if(index%2==0){//Only convert quotes not in tags.
				
				for(int i=0;i<freedata.length();i++){
					if(quoteTagCheck(i,freedata)!=null){
					
						//Opening quote tag?
						if(i==0||freedata.charAt(i-1)==' '||freedata.charAt(i-1)=='>'||freedata.charAt(i-1)=='\n'){
							temps="&#8220;";
							SB.insert(sb1,temps);
							sb1+=temps.length();
							i+=5;
						}else
						
						if(i+6>=freedata.length()||freedata.charAt(i+6)==' '||freedata.charAt(i+6)=='<'||freedata.charAt(i+6)=='\n'||isPunctuation(freedata.charAt(i+6))){
							temps="&#8221;";
							SB.insert(sb1,temps);
							sb1+=temps.length();
							i+=5;
						}else{
							SB.insert(sb1,freedata.charAt(i));
							sb1++;
						}
					}else
				
					if(freedata.charAt(i)=='\''){
						//Opening quote tag?
						if(i!=0&&i+1<freedata.length()&&isAlphaNumeric(freedata.charAt(i+1))&&isAlphaNumeric(freedata.charAt(i-1))){
							temps="&#8217;";
							SB.insert(sb1,temps);
							sb1+=temps.length();
						}else
						
						if(i==0||freedata.charAt(i-1)==' '||freedata.charAt(i-1)=='>'){
							temps="&#8216;";
							SB.insert(sb1,temps);
							sb1+=temps.length();
						}else
						
						if(i+1>=freedata.length()||freedata.charAt(i+1)==' '||freedata.charAt(i+1)=='<'||isPunctuation(freedata.charAt(i+1))	){
							temps="&#8217;";
							SB.insert(sb1,temps);
							sb1+=temps.length();
						}else{
							SB.insert(sb1,'\'');
							sb1++;
						}
					}else{
						SB.insert(sb1,freedata.charAt(i));
						sb1++;
					}
				}
			}else{
				SB.insert(sb1,freedata);
				sb1+=freedata.length();
			}
		}
		return(SB.substring(0,sb1));
	}
	
	/**
	Apply the meta information stored in the block to our tag.
	*/
	public static String applyTagMeta(TextileBlock Block){
		String opentag="";
					
		//Get the styles.
		if(Block.getStyle().length()>0){
			String style[]=Block.getStyle().split(";");
			opentag+=" style=\"";
			for(int i=0;i<style.length;i++){
				if(style[i].length()>1)
					opentag+=style[i]+";";
			}
			opentag+="\"";
		}
		
		//Language. 
		if(Block.getLanguage().length()>0)
			opentag+=" lang=\""+Block.getLanguage()+"\"";
		
		//Class.
		if(Block.getBlockClass().length()>0)
			opentag+=" class=\""+Block.getBlockClass()+"\"";
			
		//ID.
		if(Block.getID().length()>0)
			opentag+=" id=\""+Block.getID()+"\"";

		return(opentag);
	}
	
	/**
	This method (and the following helper functions) allow us to parse several types of
	links in one pass of the data. Acronym, Footnote, Link, Image, Linked Image.
	*/
	private ArrayList Links=new ArrayList();

	private int fok[]={0,2,0};
	private String parsefoot="";
	
	private int lok[]={0,4,0};
	private String parselink="";
	
	private int iok[]={0,4,0};
	private String parseimage="";
	
	private int aok[]={0,3,0};
	private String parseacronym="";
	
	public ArrayList parseLinks(String content,int size){//This function has been modified.
		return(Links);
	}
	
	public void resetLinks(){
		Links=new ArrayList();
	}
	
	//This allows us to peform the same function while in another pass through the data.
	private char lastc=' ';
	public boolean parseLinks(char c,int i,String content,boolean backtracking){
		boolean backtrack=false;//Backtracking is sometimes necessary to ensure that we don't miss a starting token.
		if(!backtracking){
			fok[0]=footOK(fok[0],c,fok[2]);
			
			//Parse footnote.
			if(fok[0]==0)
				fok[2]=0;
			else
				fok[2]++;
				
			if(fok[0]==0)
				parsefoot="";
			if(fok[0]==1&&c!='['){
				parsefoot+=c;
			}
				
			if(fok[0]==fok[1]){
				Links.add(new TextileLink(TextileLink.FOOTNOTE,i,parsefoot,12,this));
				fok[0]=0;
				parsefoot="";
			}
			
			//Parse for image.
			if(iok[0]==2&&imageOK(iok[0],c)==0)
				Links.add(new TextileLink(TextileLink.IMAGE,i,parseimage,12,this));
			iok[0]=imageOK(iok[0],c);
			
			if(iok[0]==1&&i>0&&!(content.charAt(i-1)==' '||content.charAt(i-1)=='|')&&content.charAt(i)=='!')
				iok[0]=0;
			
			if(iok[0]==0){
				parseimage="";
			}
			if(iok[0]>0){
				parseimage+=c;
			}
				
			if(iok[0]==iok[1]){
				Links.add(new TextileLink(TextileLink.IMAGE,i,parseimage,12,this));
				iok[0]=0;
				parseimage="";
			}
			
			//Parse for acronym.
			aok[0]=acronymOK(aok[0],c,aok[2]);
			
			if(aok[0]==0){
				aok[2]=0;
				parseacronym="";
			}
			if(aok[0]>0){
				aok[2]++;
				parseacronym+=c;
			}
				
			if(aok[0]==aok[1]){
				Links.add(new TextileLink(TextileLink.ACRONYM,i,parseacronym,12,this));
				aok[0]=0;
				parseacronym="";
			}
		
		}
		//Parse for iink.
		int initial=lok[0];
		lok[0]=linkOK(lok[0],c);
		if(lok[0]==2&&lastc!='t')
			lok[0]--;
		
		//Check whether we need to peform backtracking.
		if(lok[0]!=initial&&lok[0]==0){
			backtrack=true;
		}
		
		if(lok[0]==0){
			parselink="";
		}
		if(lok[0]>0){
			parselink+=c;
		}
			
		if(lok[0]==lok[1]){
			if(ExternalParser==null){
				TextileLink LI=new TextileLink(TextileLink.LINK,i,parselink,12,this);
				LI.setExternalParser(this);
				Links.add(LI);
			}else{
				TextileLink LI=new TextileLink();
				LI.setExternalParser(ExternalParser);
				LI.process(TextileLink.LINK,i,parselink,12,this);
				Links.add(LI);
			}
			lok[0]=0;
			parselink="";
		}
		lastc=c;
		return(backtrack);
	}
	
	//Checks whether the parsing of a footer link is going OK.
	protected int footOK(int phase,char c,int parsed){
		if(phase==0&&c=='['){
			phase++;
			return(phase);
		}
		if(phase==1&&c==']'&&parsed>1){
			phase++;
			return(phase);
		}
		if(phase==1&&!(c>='0'&&c<='9')){
			return(0);
		}
		return(phase);
	}

	//Checks whether the parsing of a link is going OK.
	protected int linkOK(int phase,char c){
		if(phase==0&&c==';'){
			phase++;
			return(phase);
		}
		if(phase==1&&c==';'){
			phase++;
			return(phase);
		}
		if(phase==2&&c==':'){
			phase++;
			return(phase);
		}
		if(phase==3&&(c==' '||c=='\n')){
			phase++;
			return(phase);
		}

		if((phase==2&&c!=':')||(phase!=3&&!validInURL(c))){
			return(0);
		}
		return(phase);
	}
	
	//Checks whether the parsing of an image link is going OK.
	protected int imageOK(int phase,char c){
		if(phase==0&&c=='!'){
			phase++;
			return(phase);
		}
		if(phase==1&&c=='!'){
			phase++;
			return(phase);
		}
		if(phase==2&&c==':'){
			phase++;
			return(phase);
		}
		if(phase==3&&(c==' '||c=='\n')){
			phase++;
			return(phase);
		}

		if((phase==2&&c!=':')||(phase!=3&&!validInURL(c))){
			return(0);
		}
		return(phase);
	}
	
	//Checks whether the parsing of an acronym link is going OK.
	protected int acronymOK(int phase,char c,int tokenCount){
		if(phase==0&&upperCaseLetter(c)){
			phase++;
			return(phase);
		}
		if(phase==1&&c=='('){
			if(tokenCount>1)
				phase++;
			return(phase);
		}
		if(phase==2&&c==')'){
			phase++;
			return(phase);
		}
		
		if((phase==1&&!(upperCaseLetter(c)||isNumeric(c)))||(phase==3&&c=='\n')){
			return(0);
		}
		
		return(phase);
	}
	
	
	/**
	Can this character be used as part of an URL?
	*/
	protected boolean validInURL(char c){
		char special[]={'+','-','=','.','_','/','*','(',')',',','@','\'','$',':',';','&','!',' ','?','%','#'};
		if((c>='a'&&c<='z')||(c>='A'&&c<='Z')||(c>='0'&&c<='9'))
			return(true);
		for(int i=0;i<special.length;i++)
			if(c==special[i])
				return(true);
				
		return(false);
	}
	
	/**
	Is this charcter an uppercase letter?
	*/
	public boolean upperCaseLetter(char c){
		if((c>='A'&&c<='Z'))
			return(true);
		return(false);
	}
	
	/**
	Is this charcter a letter.
	*/
	public boolean isAlpha(char c){
		if((c>='a'&&c<='z')||(c>='A'&&c<='Z'))
			return(true);
		return(false);
	}
	
	/**
	Is this character numeric?
	*/
	public boolean isNumeric(char c){
		if((c>='0'&&c<='9'))
			return(true);
		return(false);
	}
	
	/**
	Is this character valid to start parsing a Textile tag at?
	*/
	protected boolean validTagStart(char c){
		char special[]={'.','*','-','+','@','%',';',' ','!',(char)127};
		for(int i=0;i<special.length;i++)
			if(c==special[i])
				return(true);
		
		return(false);
	}
	
	/**
	Check whether the character provided is punctuation.
	*/
	public boolean isPunctuation(char c){
		char special[]={'.',',','!',';',':','?',')'};
		for(int i=0;i<special.length;i++)
			if(c==special[i])
				return(true);
		return(false);
	}
	
	/**
	Check whether the character provided is alpha numeric.
	*/
	public boolean isAlphaNumeric(char c){
		if((c>='a'&&c<='z')||(c>='A'&&c<='Z')||(c>='0'&&c<='9'))
			return(true);
		return(false);
	}
	
	/**
	Search the document for any link Aliases. [ALIAS]http:// www.somelink.com
	Should be performed on the central parsing instance... so that the links
	can be found by different threads.
	*/
	public String findLinkAliases(String freedata){

		ArrayList DataList=getParseList(freedata);
		String name="";
		String url="";
		String parse="";
		boolean on=false;
		int step=0;
		
		StringBuffer SB=new StringBuffer(freedata.length());
		StringBuffer SB2=new StringBuffer(freedata.length());
		int iii=0;
		
		for(int index=0;index<DataList.size();index++){
			SB2.insert(iii,(String)DataList.get(index));
			int ii=0;
			if(index%2==0){
				for(int i=0;i<((String)DataList.get(index)).length();i++){
					char c=SB2.charAt(i+iii);
					if(c=='['&&step==0){
						step++;
					}
					if(c==']'&&step==1){
						step++;
					}
					if((c==' '||c=='<'||c=='\n'||i==((String)DataList.get(index)).length()-1)&&step==2){
						try{
							if(url.length()>0&&name.length()>0){
								URL test=new URL(url.substring(1,url.length()));
								//We've found a link register it.
								String Link[]=new String[2];
								Link[0]=name.substring(1,name.length());
								Link[1]=url.substring(1,url.length());
								addDestination(Link);
								url="";
								name="";
								step=0;
							}else{
								throw(new Exception(""));
							}
						}catch(Exception e){
							SB.insert(ii,name);
							ii+=name.length();
							SB.insert(ii,url);
							ii+=url.length();
							step=0;
							url="";
							name="";
						}
					}
					
					if(step==0){
						SB.insert(ii,c);
						ii++;
					}
					if(step==1){
						name+=c;
					}
					if(step==2){
						url+=c;
					}
					
					//Alias can't have space.
					if(step==1&&c==' '){
						SB.insert(ii,name);
						ii+=name.length();
						step=0;
						name="";
					}
				}
				SB.insert(ii,name);
				ii+=name.length();
				SB.insert(ii,url);
				ii+=url.length();
			}
			
			if(ii>0){
				SB2.insert(iii,SB.substring(0,ii));
				iii+=ii;
			}else{
				iii+=((String)DataList.get(index)).length();
			}
		}
		return(SB2.substring(0,iii));
	}
}