/**

Description: Used to generate chunks in the PDF parser based on the current
tags activated..
Benjamin E. Coe. 2007

*/
package com.plink.plextile;
import com.plink.plextile.util.*;
import com.lowagie.text.*;
import java.util.*;
import java.awt.Color;

//Used to represent a textile Chunk, for use by the PDF.
public class TextileList extends AbstractTextileChunk{

	//Data.
	private TextileBlock TB=null;//Meta information for this list.
	private TextileParser MyTextileParser=null;//Central textile parser for fetching information.
	private int start=0;//Starting index of this list.
	private int stop=0;//Stopping position for this list.
	private int depth=0;//Current depth of list.
	private int type=NUMERIC;//Type of list NUMERIC or BULLET
	public static final int NUMERIC=0;
	public static final int BULLET=1;
	public static final String Tag[]={"ol","ul"};//Tags represented by constants.
	
	private ArrayList Chunks=new ArrayList();//ArrayList of elements in list.
	private boolean handled=false;//Has this list been handled by a block renderer?
	
	//This internal class represents an element in a list.
	private class ListElement{
		private ArrayList Chunks=new ArrayList();//List of the chunks within this list element.
		private int depth=0;//Depth of list element.
		private int start=0;//Starting index.
		private int stop=0;//Stopping index.
		private int type=0;//Numeric or Bullet.
		
		//Constructor.
		public ListElement(int depth,int start,int stop,int type){
			this.depth=depth;
			this.start=start;
			this.stop=stop;
			this.type=type;
		}
		
		//Add a chunk to this lists element array.
		public void setData(AbstractTextileChunk Chunk){
			Chunks.add(Chunk);
		}
		
		//Return depth of element..
		public int getDepth(){
			return(depth);
		}
		
		//Return an array list of chunks inside list element.
		public ArrayList getData(){
			return(Chunks);
		}
		
		//Return starting index.
		public int getStart(){
			return(start);
		}
		
		//Return stopping index.
		public int getStop(){
			return(stop);
		}
		
		//Bulleted or Numeric.
		public int getType(){
			return(type);
		}
	}
	
	//Constrructor.
	public TextileList(TextileBlock TB,TextileParser MyTextileParser,int type,int depth){
		this.depth=depth;
		this.type=type;
		if(TB!=null)
			this.TB=TB;
		else
			TB=new TextileBlock();
		this.MyTextileParser=MyTextileParser;
	}

	/**
	Set the starting and stopping position within the raw data.
	*/
	public void setStart(int start){
		this.start=start;
	}
	
	/*
	Set the stopping position within the raw data.
	*/
	public void setStop(int stop){
		this.stop=stop;
	}
	
	/**
	Insert a new list element using the provided values to construct it.
	*/
	public void insertElement(int depth,int start,int stop,int type){
		Chunks.add(new ListElement(depth,start,stop,type));
	}
	
	/**
	Add a chunk (element) to add to a specific cell in the list
	based on the position.
	*/
	public void addChunk(int position,AbstractTextileChunk Chunk){
		for(int i=0;i<Chunks.size();i++){
			ListElement temp=(ListElement)Chunks.get(i);
			if(position>=temp.getStart()&&position<=temp.getStop()){
				temp.setData(Chunk);
				break;
			}
		}
	}
	
	/**
	Check whether a given position is in this list.
	*/
	public boolean inList(int position){
		for(int i=0;i<Chunks.size();i++){
			ListElement Chunk=(ListElement)Chunks.get(i);
			if(i==0&&position==Chunk.getStart())
				return(false);
			if(position>=Chunk.getStart()&&position<=Chunk.getStop()){
				return(true);
			}
		}
		return(false);
	}
	
	/**
	Return whether the given position should be treated as an event.
	Generate HTML for a text parser.
	*/
	private int HTMLDepth=0;//Keep track of how nested the list is.
	private ArrayList DepthTag=new ArrayList();//Keep track of the HTML tags at each depth.
	
	public int isEvent(int position){
		currentHTML="";
		
		//Is a list starting?
		if(position==getStart()){
			HTMLDepth++;
			currentHTML="<"+Tag[type]+" "+MyTextileParser.applyTagMeta(TB)+"><li>";
			DepthTag.add(Tag[type]);
			for(int i=0;i<depth;i++){
				HTMLDepth++;
				DepthTag.add(Tag[type]);
				currentHTML+="<"+Tag[type]+"><li>";
			}
			
			if(type==NUMERIC)
				return(BlockRenderer.NUMERIC_LIST_START);
			else
				return(BlockRenderer.BULLETED_LIST_START);
		}
			
		//Is a list stopping?
		if(position==getStop()){
			for(int i=0;i<HTMLDepth;i++)
				currentHTML+="</li></"+(String)DepthTag.get(i)+">";
			HTMLDepth=0;
			if(type==NUMERIC){
				return(BlockRenderer.NUMERIC_LIST_END);
			}else{
				return(BlockRenderer.BULLETED_LIST_END);
			}
		}
			
		//Check for starting blocks.
		for(int i=0;i<Chunks.size();i++){
			ListElement DM=(ListElement)Chunks.get(i);
			
			//Check whether the list has either increased or decreased in depth.
			if(DM.getStart()==position){
				ListElement DMP=(ListElement)Chunks.get(i-1);
				if(DM.getDepth()<DMP.getDepth()){
					for(int ii=0;ii<DMP.getDepth()-DM.getDepth();ii++){
							HTMLDepth--;
							String tag=(String)DepthTag.get(DepthTag.size()-1);
							currentHTML+="</li></"+tag+"></li><li>";
							DepthTag.remove(DepthTag.size()-1);
					}
				}
				
				if(DM.getDepth()>DMP.getDepth()){
					for(int ii=0;ii<DM.getDepth()-DMP.getDepth();ii++){
						currentHTML+="<"+Tag[DM.getType()]+"><li>";
						DepthTag.add(Tag[DM.getType()]);
						HTMLDepth++;
					}
				}
				
				if(currentHTML.length()==0)
						currentHTML="</li><li>";
	
			}
			
			//A list entry has ended generate an event.
			if(DM.getStop()==position)
				return(BlockRenderer.ENTRY_END);
			
			//A list entry has begun generate an event.
			if(DM.getStart()==position)	
				return(BlockRenderer.ENTRY_START);
		}
		
		return(-1);
	}
	
	/**
	Has this Chunk been dealt with by a parent yet.
	*/
	public boolean isHandled(){
		return(handled);
	}
	
	/**
	Set whether the chunk has been handled.
	*/
	public void setHandled(boolean handled){
		this.handled=handled;
	}
	
	/**
	Start position of this list.
	*/
	public int getStart(){
		return(start);
	}
	
	/**
	Return stop position of list.
	*/
	public int getStop(){
		return(stop);
	}
	
	//Generate Chunk based on tags.
	public Element getChunk(){ 
	
		com.lowagie.text.List list = null;
		if(type==BULLET)
			list = new com.lowagie.text.List(false);
		else
			list = new com.lowagie.text.List(true);
			
		Font f = null;
				
		ArrayList Lists=new ArrayList();
		Lists.add(list);
		ArrayList DepthList=new ArrayList();
		DepthList.add(new Integer(depth));
		
		//Recursively build a PDF list element.
		for(int i=0;i<Chunks.size();i++){
		
			ListItem createMe=new ListItem();
			ListElement DM=(ListElement)Chunks.get(i);
			ArrayList AL=DM.getData();
			if(AL!=null)
				for(int ii=0;ii<AL.size();ii++){
					AbstractTextileChunk temp=(AbstractTextileChunk)AL.get(ii);
					if(temp.getChunk()!=null){
						if(temp.getBlock()!=null){
							TB.merge(temp.getBlock());
						}
					
						createMe.add(temp.getChunk());
					}
				}
			
			int check=(Integer)DepthList.get(DepthList.size()-1);
			
			if(DM.getDepth()>check){
			
				com.lowagie.text.List NL = null;
				if(DM.getType()==BULLET)
					NL = new com.lowagie.text.List(false);
				else
					NL = new com.lowagie.text.List(true);
				NL.add(createMe);

				com.lowagie.text.List TL=(com.lowagie.text.List)Lists.get(DepthList.size()-1);
				TL.add(NL);
				NL.setIndentationLeft(DM.getDepth()*10);
				
				DepthList.add(new Integer(DM.getDepth()));
				Lists.add(NL);
			}else if(DM.getDepth()<check){
		
				
				int insert=DepthList.size()-1;
				while(insert>=0){
					int temp=(Integer)DepthList.get(insert);
					if(temp==DM.getDepth())
						break;
					insert--;
				}
				if(insert<0)
					insert=0;
				
				com.lowagie.text.List TL=(com.lowagie.text.List)Lists.get(insert);
				TL.add(createMe);
				
				DepthList.add(new Integer(DM.getDepth()));
				Lists.add(TL);
			}else{
				com.lowagie.text.List TL=(com.lowagie.text.List)Lists.get(DepthList.size()-1);
				TL.add(createMe);
				DepthList.add(new Integer(DM.getDepth()));
				Lists.add(TL);
			}
		}
		
		return((Element)Lists.get(0));
	}
	
	/**
	Stores the last HTML generated based on list position
	events.
	*/
	private String currentHTML="";
	public String getHTML(){
		return(currentHTML);
	}
	
	/**
	Return a reference to this chunks meta data block.
	*/
	public TextileBlock getBlock(){
		return(TB);
	}
	
	/**
	toString()
	*/
	public String toString(){
		return("");
	}
}