/**

Description: Represents a Link in a Textile Document.<br /><br />
[Foonote][Href][Image][Acronym]

Benjamin E. Coe. (c)2007<br />

-------------------------------<br />
This library is provided free of charge by Benjamin E. Coe and PLink. Simply
be kind and reference your use of our library.
-------------------------------<br />
caminatored
*/
package com.plink.plextile;
import com.lowagie.text.*;
import com.lowagie.text.pdf.PdfWriter;
import com.plink.plextile.util.*;
import java.util.*;
import java.io.*;
import java.awt.Color;
import java.net.URL;
import javax.imageio.ImageIO;

public class TextileLink extends  AbstractTextileChunk{
	//Constants for link type.
	public static final int LINK=0;
	public static final int IMAGE=1;
	public static final int FOOTNOTE=2;
	public static final int ACRONYM=3;
	
	private TextileParser ExternalParser=null;
	private int alignment=0;//Values follow
	
	public static final int LEFT=0;
	public static final int RIGHT=1;
	public static final int CENTER=2;
	
	private int position=0;
	private int size=0;
	private String data="";
	private String url="";
	private String title="";
	private int type;
	private int skip=0;
	private TextileParser MyTextileParser;
	private boolean LinkReference=false;
	private String style="";
	
	//Constructor.
	public TextileLink(int type,int startpos,String parseme,int size,TextileParser MyTextileParser){
		this.MyTextileParser=MyTextileParser;
		this.size=size;
		this.type=type;
		if(type==FOOTNOTE)
			footnote(parseme,startpos);
		if(type==LINK)
			link(parseme,startpos);
		if(type==IMAGE){
			image(parseme,startpos);
		}
		if(type==ACRONYM){
			acronym(parseme,startpos);
		}
	}
	
	public TextileLink(){
	
	}
	
	/**
	Process the data provided. (Used by the threaded text parser.)
	*/
	public void process(int type,int startpos,String parseme,int size,TextileParser MyTextileParser){
		this.MyTextileParser=MyTextileParser;
		this.size=size;
		this.type=type;
		if(type==FOOTNOTE)
			footnote(parseme,startpos);
		if(type==LINK)
			link(parseme,startpos);
		if(type==IMAGE){
			image(parseme,startpos);
		}
		if(type==ACRONYM){
			acronym(parseme,startpos);
		}
	}
	
	/**
	Set an external Textile Parser that contains centralized data.
	*/
	public void setExternalParser(TextileParser ExternalParser){
		this.ExternalParser=ExternalParser;
	}
	
	//Used to parse the settings for a footnote style link.
	private void footnote(String parseme,int startpos){
		url="fn"+parseme;
		data=parseme;
		position=startpos-(parseme.length()+1);
		skip=2+parseme.length()-1;
	}
	
	//Used to parse the settings for an acronym within the document.
	private void acronym(String parseme,int startpos){
		boolean switchParse=false;
		for(int i=0;i<parseme.length();i++){
			if(parseme.charAt(i)=='(')
				switchParse=true;
			if(!switchParse&&parseme.charAt(i)!='('&&parseme.charAt(i)!=')')
				data+=parseme.charAt(i);
			else if(parseme.charAt(i)!='('&&parseme.charAt(i)!=')')
				url+=parseme.charAt(i);
		}
	
		position=startpos-(parseme.length())+1;
		skip=parseme.length()-1;
	}
	
	
	//Used to parse the settings for a regular old link.
	private void link(String parseme,int startpos){
		//Fix to prevent global replace all for token.
		parseme=parseme.replaceAll("&quot;","     \"");
		if(parseme.length()>1)
			parseme="\""+parseme.substring(1,parseme.length());
	
		position=startpos-(parseme.length())-4;
		parseme=parseme+'.';
		int length=parseme.length();
		
		boolean on=false;
		while((parseme.charAt(length-1)!='.'&&parseme.charAt(length-1)!=':'&&parseme.charAt(length-1)!='/'&&parseme.charAt(length-1)!='+'&&parseme.charAt(length-1)!='='&&parseme.charAt(length-1)!='_'&&parseme.charAt(length-1)!='?'&&parseme.charAt(length-1)!='%'&&parseme.charAt(length-1)!=';')||!on){
			if((parseme.charAt(length-1)>='a'&&parseme.charAt(length-1)<='z')||(parseme.charAt(length-1)>='A'&&parseme.charAt(length-1)<='Z')||(parseme.charAt(length-1)>='0'&&parseme.charAt(length-1)<='9')){
				on=true;
			}
			
			length--;
			if(length==0)
				break;
		}
		
		//Something bad has happened.
		if(length==0){
			skip=0;
			position=-1;
			return;
		}
		//It's a remote reference.
		if(parseme.charAt(length-1)==':'){
			LinkReference=true;
		}
		
		length++;
		//Make a safe link.
		while((parseme.charAt(length-1)>='a'&&parseme.charAt(length-1)<='z')||(parseme.charAt(length-1)>='A'&&parseme.charAt(length-1)<='Z')||(parseme.charAt(length-1)>='0'&&parseme.charAt(length-1)<='9')||parseme.charAt(length-1)=='-'||parseme.charAt(length-1)=='_'||parseme.charAt(length-1)=='/'||parseme.charAt(length-1)=='~'||parseme.charAt(length-1)==';'){
			length++;
		}
		length--;
		parseme=parseme.substring(0,length);
		
		//Get the text to link.
		on=false;
		data="";
		for(int i=0;i<parseme.length();i++){
			if(on&&parseme.charAt(i)!='"')
				data+=parseme.charAt(i);
			if(parseme.charAt(i)=='"'&&!on)
				on=true;
			else if(parseme.charAt(i)=='"'&&on)
				break;
		}
		if(parseme.length()>=data.length()+3)
			url=parseme.substring(data.length()+3,parseme.length());
		
		if(data.length()>=5){
			data=data.substring(0,data.length()-5);
		}
		skip=parseme.length()+4;
		
		on=false;
		String temp="";
		for(int i=0;i<data.length();i++){
			if(!on&&data.charAt(i)!='(')
				temp+=data.charAt(i);
			else
				on=true;
				
			if(on&&data.charAt(i)!=')'&&data.charAt(i)!='(')
				title+=data.charAt(i);
		}
		data=temp;
		
		if(LinkReference){
			if(ExternalParser!=null){
				if(ExternalParser.destinationExists(url)){
					url=ExternalParser.getDestination(url);
				}else{
					url="#"+url;
				}
			}else
		
			if(MyTextileParser.destinationExists(url)){
				url=MyTextileParser.getDestination(url);
			}else{
				url="#"+url;
			}
		}
	}
	
	//Used to parse the settings for an image link.
	private void image(String parseme,int startpos){
	
		position=startpos-(parseme.length());
		parseme=parseme+'.';
		int length=parseme.length();
		
		boolean on=false;
		while((parseme.charAt(length-1)!='.'&&parseme.charAt(length-1)!=':'&&parseme.charAt(length-1)!='/'&&parseme.charAt(length-1)!='+'&&parseme.charAt(length-1)!='='&&parseme.charAt(length-1)!='?'&&parseme.charAt(length-1)!='%'&&parseme.charAt(length-1)!=';')||!on){
			if((parseme.charAt(length-1)>='a'&&parseme.charAt(length-1)<='z')||(parseme.charAt(length-1)>='A'&&parseme.charAt(length-1)<='Z')||(parseme.charAt(length-1)>='0'&&parseme.charAt(length-1)<='9')){
				on=true;
			}
			length--;
			if(length==0||parseme.charAt(length-1)=='!'){
				length=0;
				break;
			}
		}
		
		//It must just be an image.
		boolean justimage=false;
		if(length==0){
			justimage=true;
			length=parseme.length();
		}
		
		//It's a remote reference.
		if(parseme.charAt(length-1)==':'){
			LinkReference=true;
		}
		
		if(!justimage){
			//Linked image.
			position++;
			length++;
			//Make a safe link.
			while((parseme.charAt(length-1)>='a'&&parseme.charAt(length-1)<='z')||(parseme.charAt(length-1)>='A'&&parseme.charAt(length-1)<='Z')||(parseme.charAt(length-1)>='0'&&parseme.charAt(length-1)<='9')||parseme.charAt(length-1)=='-'||parseme.charAt(length-1)=='_'||parseme.charAt(length-1)=='/'||parseme.charAt(length-1)=='~'||parseme.charAt(length-1)==';'){
				length++;
			}
			length--;
			parseme=parseme.substring(0,length);
			
			//Get the text to link.
			on=false;
			data="";
			for(int i=0;i<parseme.length();i++){
				if(on&&parseme.charAt(i)!='!')
					data+=parseme.charAt(i);
				if(parseme.charAt(i)=='!'&&!on)
					on=true;
				else if(parseme.charAt(i)=='!'&&on)
					break;
			}
	
			if(parseme.length()>=data.length()+3)
				url=parseme.substring(data.length()+3,parseme.length());
			
			on=false;
			String temp="";
			for(int i=0;i<data.length();i++){
				if(!on&&data.charAt(i)!='(')
					temp+=data.charAt(i);
				else
					on=true;
					
				if(on&&data.charAt(i)!=')'&&data.charAt(i)!='(')
					title+=data.charAt(i);
			}
			data=temp;
			
			if(LinkReference){
				if(MyTextileParser.destinationExists(url)){
					url=MyTextileParser.getDestination(url);
				}else{
					url="#"+url;
				}
			}
			
			skip=parseme.length()-1;
		}else{
			//Image with no link.
			on=false;
			data="";
			for(int i=0;i<parseme.length();i++){
				if(on&&parseme.charAt(i)!='!')
					data+=parseme.charAt(i);
				if(parseme.charAt(i)=='!'&&!on)
					on=true;
				else if(parseme.charAt(i)=='!'&&on)
					break;
			}
			
			on=false;
			String temp="";
			for(int i=0;i<data.length();i++){
				if(!on&&data.charAt(i)!='(')
					temp+=data.charAt(i);
				else
					on=true;
					
				if(on&&data.charAt(i)!=')'&&data.charAt(i)!='(')
					title+=data.charAt(i);
			}
			data=temp;
			
			skip=parseme.length()-2;
		}
		
		//Get the image alignment information.
		if(data.length()>0){
			data=data.replaceAll("&gt;",">");
			data=data.replaceAll("&lt;","<");
			if(data.charAt(0)=='>'){
				alignment=RIGHT;
				data=data.substring(1,data.length());
				style="float:right";
			}else
			
			if(data.charAt(0)=='<'){
				alignment=LEFT;
				data=data.substring(1,data.length());
				style="float:left";
			}else
			
			if(data.charAt(0)=='='){
				alignment=CENTER;
				data=data.substring(1,data.length());
				style="float:center";
			}
		}
	}
	
	
	//At what position in the tagfree text is this link located.
	public int getPosition(){
		return(position);
	}
	
	//How may spaces in the document should we skip to ignore this element.
	public int getSkip(){
		return(skip);
	}
	
	/**
	Get the HTML link associated with this object.
	*/
	public String getHTML(){
	
		if(url!=null){
			//url=url.replaceAll("&amp;","&");
			url=url.replaceAll(">","\\*\\*");
			url=url.replaceAll("<","__");
			url=url.replaceAll(""+((char)127),"\\?\\?");
		}
		
		if(data!=null){
			data=data.replaceAll(">","\\*\\*");
			data=data.replaceAll("<","__");
			data=data.replaceAll(""+((char)127),"\\?\\?");
		}
		
		if(title!=null){
			title=title.replaceAll(">","\\*\\*");
			title=title.replaceAll("<","__");
			title=title.replaceAll(""+((char)127),"\\?\\?");
		}
	
		if(type==FOOTNOTE){
			String returnMe="<sup class=\"footnote\"><a href=\"#"+url+"\">"+data+"</a></sup>";
			return(returnMe);
		}
		
		if(type==ACRONYM){
			String returnMe="<acronym title=\""+url+"\">"+data+"</acronym>";
			return(returnMe);
		}
		
		if(type==LINK){
			String returnMe="";
			if(title.length()>0)
				returnMe="<a href=\""+url+"\" title=\""+title+"\">"+data+"</a>";
			else
				returnMe="<a href=\""+url+"\">"+data+"</a>";
			return(returnMe);
		}
		if(type==IMAGE){
			
			String returnMe="";
			if(url.length()>0)
				returnMe+="<a href=\""+url+"\">";
			
			if(title.length()>0)
				returnMe+="<img src=\""+data+"\" title=\""+title+"\" alt=\""+title+"\" style=\""+style+"\"/>";
			else
				returnMe+="<img src=\""+data+"\" alt=\"http://www.plink-search.com\" style=\""+style+"\"/>";
			
			if(url.length()>0)
				returnMe+="</a>";
			return(returnMe);
		}
		return("");
	}
	
	/**
	Create a PDF chunk out of this Link.
	*/
	public Element getChunk(){
		if(url!=null){
			//url=url.replaceAll("&amp;","&");
			url=url.replaceAll(">","\\*\\*");
			url=url.replaceAll("<","__");
			url=url.replaceAll(""+((char)127),"\\?\\?");
		}
		
		if(data!=null){
			data=data.replaceAll(">","\\*\\*");
			data=data.replaceAll("<","__");
			data=data.replaceAll(""+((char)127),"\\?\\?");
		}
		
		if(title!=null){
			title=title.replaceAll(">","\\*\\*");
			title=title.replaceAll("<","__");
			title=title.replaceAll(""+((char)127),"\\?\\?");
		}
	
		String tempdata=data;
		
		//If the link is of ACRONYM type just re-output the text.
		if(type!=ACRONYM){
			tempdata=MyTextileParser.finalReplacement(tempdata);
			tempdata=HTMLFilter.reconvertString(tempdata);
		}else{
			tempdata=MyTextileParser.finalReplacement(tempdata+"("+url+")");
			tempdata=HTMLFilter.reconvertString(tempdata);
		}
			
		Chunk chunk = new Chunk(tempdata); 
		Font f = new Font(Font.TIMES_ROMAN,size,Font.NORMAL);
		
		//This is an image.
		if(type==IMAGE){
			try{
				java.awt.Image awtImage = ImageIO.read(new URL(data));
				Image image = Image.getInstance(awtImage, null);
				image.scalePercent(70);
				chunk=new Chunk(image,0,0,true);
				if(url.length()>0)
					chunk.setAnchor(url);
			}catch(Exception e){
				if(url.length()>0)
					chunk.setAnchor(url);
				f.setColor(new Color(0,0,255));
				chunk.setFont(f);
			}
		}
		
		//Is this a footnote?
		if(type==FOOTNOTE){
			chunk.setTextRise(5.0f);
			chunk.setLocalGoto(url);
			boolean set=false;
			if(MyTextileParser.destinationExists(url))
				set=true;
			
			if(!set)
				chunk.setLocalDestination(url);
			f.setSize(size-4);
			chunk.setFont(f);
		}
		
		//Is this a Link.
		if(type==LINK){
			chunk.setAnchor(url);
			f.setColor(new Color(0,0,255));
			chunk.setFont(f);
		}
		return(chunk);
	}
	
	
	/**
	Return a reference to this block.
	NOT CURRENTLY APPLICABLE TO LINK CHUNK.
	*/
	public TextileBlock getBlock(){
		return(null);
	}
	
	/**
	toString();
	*/
	public String toString(){
		return("");
	}
	
	public int getType(){
		return(type);
	}
}