/**

Description: Parses a string containing Textile markup.<br /><br />

Benjamin E. Coe. (c)2007<br />

-------------------------------<br />
This library is provided free of charge by Benjamin E. Coe and PLink. Simply
be kind and reference your use of our library.
-------------------------------<br />
caminatored
*/
package com.plink.plextile;
import  com.plink.plextile.util.*;
import java.util.*;

public class TextParser{
	private TextileParser PublicTextileParser=null;
	
	public TextParser(){
		PublicTextileParser=new TextileParser();
	}
	
	/**
	Apply the block formatting settings to the block of text.
	*/
	protected String applyBlock(TextileBlock Block,String data){
		String opentag="";
		if(!Block.getTag().equals("bc"))
			opentag="<"+Block.getTag();
					
		//Get the styles.
		if(Block.getStyle().length()>0){
			String style[]=Block.getStyle().split(";");
			opentag+=" style=\"";
			for(int i=0;i<style.length;i++){
				if(style[i].length()>1)
					opentag+=style[i]+";";
			}
			opentag+="\"";
		}
		
		//Language. 
		if(Block.getLanguage().length()>0)
			opentag+=" lang=\""+Block.getLanguage()+"\"";
		
		//Class.
		if(Block.getBlockClass().length()>0)
			opentag+=" class=\""+Block.getBlockClass()+"\"";
			
		//ID.
		if(Block.getID().length()>0)
			opentag+=" id=\""+Block.getID()+"\"";

		if(Block.getTag().equals("bc")){
			String temp="<pre"+opentag+"><code "+opentag+">"+data+"</code></pre>";
			return(temp);
		}
		opentag+=">";
		String closetag="</"+Block.getTag()+">";
		return(opentag+data+closetag);
	}
	
	/**
	Parse the block of text provided. (Filter is applied.)<br />
	allowHTML dictates whether reserved HTML characters should be escaped.
	*/
	public synchronized String parseTextile(String data,boolean allowHTML){
		
		data=HTMLFilter.filter(data);//Replace dangerous characters.
		
		//Initial processing.
		data=PublicTextileParser.initialEscape(data);
		
		//Make sure that \n\n in code blocks doesn't break flow.
		data=PublicTextileParser.makeCodeSafe(data);
		
		//First parse all the aliases in the document.
		data=PublicTextileParser.findLinkAliases(data+" ");
		
		//Split the blocks based on the </p> we inserted.
		String chunks[]=data.split("(\n\n)");
		
		//Split into blocks and parse each.
		data="";
		
		ArrayList Processes=new ArrayList();
		for(int i=0;i<chunks.length;i++){
			if(Processes.size()<8){
				Processes.add(new ParseBlockThread(chunks[i],i));
			}else{
				while(Processes.size()>5){
					ParseBlockThread PBT=(ParseBlockThread)Processes.get(0);
					if(!PBT.isRunning()){
						chunks[PBT.getID()]=PBT.getData();
						Processes.remove(0);
					}
				}
				Processes.add(new ParseBlockThread(chunks[i],i));
			}
		}
		
		while(Processes.size()>0){
			ParseBlockThread PBT=(ParseBlockThread)Processes.get(0);
			if(!PBT.isRunning()){
				//If HTML is allowed in our data perform the following step.
				if(allowHTML&&!PBT.isCode()){
					String tempdata=PBT.getData();
					tempdata=tempdata.replaceAll("&amp;","&");
					tempdata=tempdata.replaceAll("&quot;","\"");
					tempdata=tempdata.replaceAll("&lt;","<");
					tempdata=tempdata.replaceAll("&gt;",">");
					tempdata=tempdata.replaceAll("<pre>","\n<pre>");
					tempdata=tempdata.replaceAll("<br /><code>","\n<code>");
					tempdata=tempdata.replaceAll("</code><br />","</code>\n");
					tempdata=tempdata.replaceAll("&#"+((int)'>')+";","&gt;");
					tempdata=tempdata.replaceAll("&#"+((int)'<')+";","&lt;");
					tempdata=tempdata.replaceAll("&#10;","\n");
					chunks[PBT.getID()]=tempdata;
				}else{
					chunks[PBT.getID()]=PBT.getData();
				}
				Processes.remove(0);
			}
		}
		
		for(int i=0;i<chunks.length;i++){
			data+=chunks[i];
		}
		
		return(data);
	}
	
	
	/**
	Parses a block of text. Only use this method if the data
	is already stored in the HTMLFilter form.
	*/
	public synchronized String parseTextileNoFilterShort(String data,boolean allowHTML){
		//Initial processing.
		data=PublicTextileParser.initialEscape(data);
		
		//Make sure that \n\n in code blocks doesn't break flow.
		data=PublicTextileParser.makeCodeSafe(data);
		
		
		//First parse all the aliases in the document.
		data=PublicTextileParser.findLinkAliases(data+" ");
		
		
		//Split the data at the </p> we inserted.
		String chunks[]=data.split("\n\n");
		int size=0;
		int chunk_count=0;
		for(int i=0;i<chunks.length;i++){
			size+=chunks[i].length();
			chunk_count++;
			if(size>=256)
				break;
		}
		String temp_chunks[]=new String[chunk_count];
		for(int i=0;i<chunk_count;i++){
			temp_chunks[i]=chunks[i];
		}
		chunks=temp_chunks;
		
		//Split into blocks and parse each.
		data="";
		
		//Create several processes to parse text in parallel.
		ArrayList Processes=new ArrayList();
		for(int i=0;i<chunks.length;i++){
			if(Processes.size()<8){
				Processes.add(new ParseBlockThread(chunks[i],i));
			}else{
				while(Processes.size()>5){
					ParseBlockThread PBT=(ParseBlockThread)Processes.get(0);
					if(!PBT.isRunning()){
						chunks[PBT.getID()]=PBT.getData();
						Processes.remove(0);
					}
				}
				Processes.add(new ParseBlockThread(chunks[i],i));
			}
		}

		while(Processes.size()>0){
			ParseBlockThread PBT=(ParseBlockThread)Processes.get(0);
			if(!PBT.isRunning()){
				//If HTML is allowed in our data perform the following step.
				if(allowHTML&&!PBT.isCode()){
					String tempdata=PBT.getData();
					tempdata=tempdata.replaceAll("&amp;","&");
					tempdata=tempdata.replaceAll("&quot;","\"");
					tempdata=tempdata.replaceAll("&lt;","<");
					tempdata=tempdata.replaceAll("&gt;",">");
					tempdata=tempdata.replaceAll("<pre>","\n<pre>");
					tempdata=tempdata.replaceAll("<br /><code>","\n<code>");
					tempdata=tempdata.replaceAll("</code><br />","</code>\n");
					tempdata=tempdata.replaceAll("&#"+((int)'>')+";","&gt;");
					tempdata=tempdata.replaceAll("&#"+((int)'<')+";","&lt;");
					tempdata=tempdata.replaceAll("&#10;","\n");
					chunks[PBT.getID()]=tempdata;
				}else{
					chunks[PBT.getID()]=PBT.getData();
				}
				Processes.remove(0);
			}
		}
		
		for(int i=0;i<chunks.length;i++){
			data+=chunks[i];
		}
		
		return(data);
	}
	
	
	
	/**
	Parses a block of text. Only use this method if the data
	is already stored in the HTMLFilter form.
	*/
	public synchronized String parseTextileNoFilter(String data,boolean allowHTML){
		//Initial processing.
		data=PublicTextileParser.initialEscape(data);
		
		//Make sure that \n\n in code blocks doesn't break flow.
		data=PublicTextileParser.makeCodeSafe(data);
		
		
		//First parse all the aliases in the document.
		data=PublicTextileParser.findLinkAliases(data+" ");
		
		
		//Split the data at the </p> we inserted.
		String chunks[]=data.split("\n\n");
		
		//Split into blocks and parse each.
		data="";
		
		//Create several processes to parse text in parallel.
		ArrayList Processes=new ArrayList();
		for(int i=0;i<chunks.length;i++){
			if(Processes.size()<8){
				Processes.add(new ParseBlockThread(chunks[i],i));
			}else{
				while(Processes.size()>5){
					ParseBlockThread PBT=(ParseBlockThread)Processes.get(0);
					if(!PBT.isRunning()){
						chunks[PBT.getID()]=PBT.getData();
						Processes.remove(0);
					}
				}
				Processes.add(new ParseBlockThread(chunks[i],i));
			}
		}

		while(Processes.size()>0){
			ParseBlockThread PBT=(ParseBlockThread)Processes.get(0);
			if(!PBT.isRunning()){
				//If HTML is allowed in our data perform the following step.
				if(allowHTML&&!PBT.isCode()){
					String tempdata=PBT.getData();
					tempdata=tempdata.replaceAll("&amp;","&");
					tempdata=tempdata.replaceAll("&quot;","\"");
					tempdata=tempdata.replaceAll("&lt;","<");
					tempdata=tempdata.replaceAll("&gt;",">");
					tempdata=tempdata.replaceAll("<pre>","\n<pre>");
					tempdata=tempdata.replaceAll("<br /><code>","\n<code>");
					tempdata=tempdata.replaceAll("</code><br />","</code>\n");
					tempdata=tempdata.replaceAll("&#"+((int)'>')+";","&gt;");
					tempdata=tempdata.replaceAll("&#"+((int)'<')+";","&lt;");
					tempdata=tempdata.replaceAll("&#10;","\n");
					chunks[PBT.getID()]=tempdata;
				}else{
					chunks[PBT.getID()]=PBT.getData();
				}
				Processes.remove(0);
			}
		}
		
		for(int i=0;i<chunks.length;i++){
			data+=chunks[i];
		}
		
		return(data);
	}
	
	/**
	A thread class for peforming threaded block parsing...
	great for larger documents.
	*/
	private class ParseBlockThread implements Runnable{
		private String data;
		private Thread MyThread=null;
		private int id=0;
		private TextileParser MyTextileParser=new TextileParser();
		private boolean running=true;
		private boolean code=false;
		
		//Constructor..
		public ParseBlockThread(String data,int id){
			MyTextileParser.setExternalParser(PublicTextileParser);
			this.id=id;
			this.data=data;
			MyThread=new Thread(this);
			MyThread.start();
		}
		
		public synchronized void run(){
		
			String parseMe=data;
			//Get the block information.
			MyTextileParser.resetSkip();
			TextileBlock Block=new TextileBlock(parseMe);
			parseMe=Block.getData()+" ";
			
			/*BlockRenderer Factory=new RendererFactory(Block,MyTextileParser);
			BlockRenderer MyBlockRenderer=Factory.blockRendererFactory(parseMe,true);*/
			
			BlockRenderer MyBlockRenderer=(BlockRenderer)(new ComplexBlockRenderer(Block,MyTextileParser,true));
			
			//Is it a code block?
			if(Block.getTag().equals("bc")){
				data=applyBlock(Block,parseMe);
				code=true;
				running=false;
				MyThread=null;
			}else{
			
				
				int tagon[]=new int[11];//Toggle tags on and off.
				//Our block tags.
				String tags[]={"strong","em","del","ins","sup","sub","code","span","b","i","cite"};//Html mapping from special characters.
			
				ArrayList Rules=MyTextileParser.ParseTextile(parseMe,(ComplexBlockRenderer)MyBlockRenderer);//Get the tagging positions in the text.
			
				String freedata=MyTextileParser.getTagFreeData();//Get the processed data.
				
				//freedata=MyBlockRenderer.registerData(freedata);
				String returnMe="";
				
				//Get our Links list.
				ArrayList Links=MyTextileParser.parseLinks(freedata,0);
				
				int skip=0;
				boolean BlockApplied=false;
				
				int sb1=0;
				StringBuffer SB=new StringBuffer((int)(freedata.length()*1.5));
				
				boolean inTable=false;//Are we in a table element?
				boolean hasHTML=false;
				String eventHTML="";//Fix a bug regarding the events looking ahead.
				for(int i=0;i<freedata.length();i++){
					
					//Handles a special case which crops up when parsing table tokens.
					if(inTable)
					if(MyBlockRenderer.isEvent(i+1)){
						eventHTML=MyBlockRenderer.getHTML(0);
						hasHTML=true;
						int check=MyBlockRenderer.lastEvent();
						if(check==BlockRenderer.ROW_END||check==BlockRenderer.TABLE_END){
							skip++;
						}
					}
					
					//Check for incoming events from our more complex objects during the parsing process.
					if(MyBlockRenderer.isEvent(i)){
						
						String data=MyBlockRenderer.getHTML(0)+'\n';
						int check=MyBlockRenderer.lastEvent();
						
						//A Numeric list started.
						if(check==BlockRenderer.NUMERIC_LIST_START){
							if(!BlockApplied){
								if(i>0)
									returnMe=applyBlock(Block,MyTextileParser.quoteConversion(returnMe));
								BlockApplied=true;
							}
							
							SB.insert(sb1,data);
							sb1+=data.length();
						}else
						
						if(check==BlockRenderer.BULLETED_LIST_START){//A Bulleted list started.
							if(!BlockApplied){
								if(i>0)
									returnMe=applyBlock(Block,MyTextileParser.quoteConversion(returnMe));
								BlockApplied=true;
							}
							
							SB.insert(sb1,data);
							sb1+=data.length();
							
						}else if(check==BlockRenderer.ENTRY_START||check==BlockRenderer.BULLETED_LIST_END||check==BlockRenderer.NUMERIC_LIST_END){
							//Listen for specific list events.
							SB.insert(sb1,data);
							sb1+=data.length();		
						}
						
						//Table events.
						if(check==BlockRenderer.TABLE_START){//A table started.
							inTable=true;
							if(!BlockApplied){
								if(i>0)
									returnMe=applyBlock(Block,MyTextileParser.quoteConversion(returnMe));
								BlockApplied=true;
							}
							SB.insert(sb1,data);
							sb1+=data.length();	
						}else if(check==BlockRenderer.TABLE_END||check==BlockRenderer.ROW_END||check==BlockRenderer.CELL_END){
						
							//A table event occured.
							if(hasHTML)
								data=eventHTML;
						
							SB.insert(sb1,data);
							sb1+=data.length();	
						}
						
						hasHTML=false;
					}
				
					//Check whether the current position in the data should not be coppied to our buffer.
					if(MyTextileParser.getSkip(i)>0)
						skip+=MyTextileParser.getSkip(i);
					
					//Check for ending positions of tags.
					for(int ii=Rules.size()-1;ii>=0;ii--){
						int ranges[][]=(int[][])Rules.get(ii);
						if(ranges[0][1]==i){
							if(tagon[ranges[1][0]]>0)
								tagon[ranges[1][0]]--;
							if(tagon[ranges[1][0]]==0){
								String temps="</"+tags[ranges[1][0]]+">";
								SB.insert(sb1,temps);
								sb1+=temps.length();
								skip=1;
							}
						}
					}
					
					//Check for starting positions of tags.
					for(int ii=0;ii<Rules.size();ii++){
						int ranges[][]=(int[][])Rules.get(ii);
						if(ranges[0][0]==i){
							sb1--;//Was a substring.
							if(sb1<0)
								sb1=0;
							
							TagMetaData TMD=null;
							//Check for tag meta data [Language](Class){Style}
							if((TMD=MyTextileParser.hasAttrib(i))!=null){
								skip=TMD.getSkip();
								String temps="<"+tags[ranges[1][0]]+" "+MyTextileParser.applyTagMeta(TMD.getBlock())+">";
								SB.insert(sb1,temps);
								sb1+=temps.length();
							}else{
								String temps="<"+tags[ranges[1][0]]+">";
								SB.insert(sb1,temps);
								sb1+=temps.length();
							}
							tagon[ranges[1][0]]++;
						}
					}
					
					//We do a temporary replacement of these characters during the parsing step. Here we change it back.
					if(skip==0){
						char c=freedata.charAt(i);
						if(c=='\n'){
							SB.insert(sb1,"<br />\n");
							sb1+=6;
						}else if(c=='>'){
							SB.insert(sb1,"**");
							sb1+=2;
						}else if(c=='<'){
							SB.insert(sb1,"__");
							sb1+=2;
						}else if(c==(char)127){
							SB.insert(sb1,"??");
							sb1+=2;
						}else{
							SB.insert(sb1,c);
							sb1+=1;
						}
					}
					if(skip>0)
						skip--;
						
					//Does a link start at this portion of text?
					for(int ii=0;ii<Links.size();ii++){
						TextileLink LI=(TextileLink)Links.get(ii);
						if(LI.getPosition()==i){
							if(SB.charAt(sb1-1)!='>'){
							  sb1--;
							}
							skip=LI.getSkip();
							SB.insert(sb1,LI.getHTML());
							sb1+=LI.getHTML().length();
						}
					}
				}
				SB.insert(sb1,'\n');//Add a newline character.
				sb1++;
				returnMe=SB.substring(0,sb1);//Copy the StringBuffer to returnMe.
				
				
				//Apply quote conversion and return data.
				if(!BlockApplied){
					data=applyBlock(Block,MyTextileParser.quoteConversion(returnMe));
				}else
					data=MyTextileParser.quoteConversion(returnMe);
					
				running=false;
				MyThread=null;
			}
		}
		
		//Get the ID associated with this thread.
		public int getID(){
			return(id);
		}
		
		//Get the data that has been parsed.
		public String getData(){
			return(data);
		}
		
		//Is this a code block?
		public boolean isCode(){
			return(code);
		}
		
		//Is the thread done running?
		public boolean isRunning(){
			return(running);
		}
	}
}